import Foundation

public protocol TemplateInsertable {
    var stringValue: String { get }
}

extension String: TemplateInsertable {
    public var stringValue: String { return self }
}

extension Int: TemplateInsertable {
    public var stringValue: String { return "\(self)" }
}

public struct Template<T> {

    private var template: String = ""
    private var keyPaths: [(offset: String.IndexDistance, keyPath: PartialKeyPath<T>)] = []

    mutating func append<U: TemplateInsertable>(keyPath: KeyPath<T, U>) {
        keyPaths.append((offset: template.count, keyPath: keyPath))
    }

    mutating func append(string: TemplateInsertable) {
        template.append(string.stringValue)
    }

    public func fill(with value: T) -> String {
        var fullString = template
        var indexOffset: String.IndexDistance = 0
        for item in keyPaths {
            let stringValue = (value[keyPath: item.keyPath] as! TemplateInsertable).stringValue
            let insertionIndex = fullString.index(fullString.startIndex, offsetBy: item.offset + indexOffset)
            fullString.insert(contentsOf: stringValue, at: insertionIndex)
            indexOffset += stringValue.count
        }

        return fullString
    }
}

extension Template: ExpressibleByStringLiteral {
    public typealias StringLiteralType = String

    public init(stringLiteral: String) {
        append(string: stringLiteral)
    }
}

extension Template: ExpressibleByStringInterpolation {

    public init(stringInterpolation: StringInterpolation) {
        self = stringInterpolation.template
    }

    public struct StringInterpolation: StringInterpolationProtocol {

        fileprivate var template: Template<T>

        public init(literalCapacity: Int, interpolationCount: Int) {
            template = Template<T>()
        }

        mutating public func appendLiteral(_ literal: String) {
            template.append(string: literal)
        }

        mutating public func appendInterpolation<U: TemplateInsertable>(keyPath: KeyPath<T, U>) {
            template.append(keyPath: keyPath)
        }
    }
}

struct Values {
    let username: String
    let id: Int
}

let template: Template<Values> = "/users/\(keyPath: \.username)/\(keyPath: \.id)"
let fullString = template.fill(with: Values(username: "zac", id: 42))
